<?php


require_once __DIR__ . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . 'MigrationBase.php';
require_once TONIDO_CLOUD_ROOT_DIR . DIRECTORY_SEPARATOR . 'src' . DIRECTORY_SEPARATOR . 'runtime.php';

use CodeLathe\Application\ContainerFacade;
use CodeLathe\Core\Common\Utility\Utility;
use CodeLathe\Core\Infrastructure\Settings\SettingsDataStore;
use CodeLathe\Core\Infrastructure\Settings\SettingsManager;
use CodeLathe\Core\Infrastructure\Settings\SiteSetting;
use CodeLathe\Core\Infrastructure\Storage\Cryptfs\Crypt;
use CodeLathe\Core\Infrastructure\Storage\Cryptfs\CryptHelper;
use CodeLathe\Core\Infrastructure\Storage\CryptFSStoreImpl;
use CodeLathe\Core\Infrastructure\Storage\DataStore;
use CodeLathe\Core\Infrastructure\Storage\StorageConfig;


class FipsMigration extends MigrationBase
{
    /**
     * @var SettingsDataStore
     */
    private $settingsDataStore;

    /**
     * @var SettingsManager
     */
    private $settingsManager;

    /**
     * @var DataStore
     */
    private $dataStore;

    /**
     * @var CryptFSStoreImpl
     */
    private $storageImpl;


    public function __construct()
    {
        $this->settingsDataStore = SettingsDataStore::getInstance();
        $this->settingsManager = ContainerFacade::get(SettingsManager::class);
    }

    public function getTitle(): string
    {
        return 'FIPS Migration for Encrypted Storage';
    }

    protected function getSupportedVersions(): array
    {
        return ['22.1', '23.1', '23.232',];
    }

    public function migrate(): void
    {
        if (!$this->isDatabaseInitialized()) {
            throw new \Exception('Database Error: Database is not Initialized, please check if database is running.');
        }

        $sites = $this->getAllSites();

        if (empty($sites)) {
            throw new \Exception('No sites found to migrate.');
        }

        $siteKeys = [];
        $errors = [];
        foreach ($sites as $siteSettings) {
            try {
                // it's a dry run, just checks if we can get raw File Key for each website that has enabled Encrypted Storage
                $this->populateFileKeyForSite($siteKeys, $siteSettings);
            } catch (\Exception $e) {
                $errors[$siteSettings->getSiteHost()] = $e->getMessage();
            }
        }

        if (!empty($errors)) {
            throw new \Exception(sprintf("\nPre-Upgrade stopped, actions required:\n\n%s", implode("\n", $errors)));
        }

        if (empty($siteKeys)) {
            echo "No Sites with Encrypted Storage has been found, backup of File Key is not needed!\n";
            return;
        }

        foreach ($siteKeys as $siteHost => $rawFileKey) {
            $result = $this->backupRawFileKey($siteHost, $rawFileKey);
            echo sprintf("File Key backup for host `%s` %s \n", $siteHost, $result ? "SUCCEED" : "FAILED");
        }
    }

    private function populateFileKeyForSite(array &$siteKeys, SiteSetting $siteSettings): void
    {
        $isMultiTenant = defined("TONIDOCLOUD_MULTISITE_ENABLE") && (int)constant('TONIDOCLOUD_MULTISITE_ENABLE') === 1;

        $siteHost = $siteSettings->getSiteHost();

        if ($siteHost !== '*' && !$isMultiTenant) {
            throw new \Exception('Multiple sites settings has been found but Multisite is not enabled');
        }

        if ($siteHost !== '*') {
            // switch site host only for not default website
            setSiteHostName($siteHost);
        }

        $this->settingsManager = ContainerFacade::get(SettingsManager::class);
        $this->dataStore = ContainerFacade::get(DataStore::class);
        $this->storageImpl = StorageConfig::getStorageImpl();

        if (!$this->isLanManagedStorage()) {
            echo sprintf("Skipping host `%s` [NOT LAN STORAGE]\n", $siteHost);
            return;
        }

        if (!$this->isEncryptedStorageEnabled()) {
            echo sprintf("Skipping host `%s` [NOT ENCRYPTED STORAGE]\n", $siteHost);
            return;
        }

        $cryptCollection = StorageConfig::getInstance()->getDBConnection()->selectCollection('cryptfs');
        $keys = $cryptCollection->findOne(['name' => 'keys']);

        if (isset($keys['value']['backupFileKey'])) {
            // for HA environments the pre-upgrade script was already executed and raw key backup-ed
            echo sprintf("Skipping host `%s` [FILE KEY ALREADY BACKED-UP]\n", $siteHost);
            return;
        }

        if (isset($keys['value']['fipsCompliant'])) {
            // for HA environments the pre-upgrade script was already executed and raw key backup-ed and re-encrypted
            echo sprintf("Skipping host `%s` [FILE KEY ALREADY RE-ENCRYPTED]\n", $siteHost);
            return;
        }

        if (!$this->isDataStoreReady()) {
            throw new \Exception(sprintf('Encrypted storage not active for host: `%s`. Please activate Encrypted Storage for that host.', $siteHost));
        }

        $rawFileKey = $this->getRawFileKeyFromMemcache();

        if (empty($rawFileKey)) {
            throw new \Exception(sprintf('Encrypted storage not active for host: `%s`. Please activate Encrypted Storage for that host.', $siteHost));
        }

        if (!$this->verifyFileKeyForSite($siteHost)) {
            throw new \Exception(sprintf('File Key validation failed for host: `%s`. Please reach out to the FileCloud support team at support@filecloud.com', $siteHost));
        }

        $siteKeys[$siteHost] = $rawFileKey;
        Crypt::destroyInstance();
    }

    private function verifyFileKeyForSite($siteHost): bool
    {
        $pipeline = [
            [
                '$match' => [
                    'complete' => '1',
                    'type' => 'file', // we can validate only PDF files if they are valid after decryption
                    'ext' => 'pdf'
                ]
            ],
            [
                '$sample' => [
                    'size' => 5
                ]
            ]
        ];

        $cursor = $this->dataStore->getItemsByCriteria($pipeline);

        $fileKeyVerificationValid = 0;
        foreach ($cursor as $fileItem) {
            $filephypath = $fileItem['storedpath'];

            $srcLocalPath = $this->storageImpl->getLocalPathForStoreFile($filephypath);
            $ext = substr($srcLocalPath, -3);

            if ($ext === 'dat') {
                echo sprintf("Encrypted Storage is enabled but files are not encrypted for host: `%s`.\n", $siteHost);
                return true;
            }

            if (!file_exists($srcLocalPath)) {
                continue;
            }

            $filecontent = file_get_contents($srcLocalPath);
            /**
             * we can validate only PDF files if they are valid after decryption,
             * with `%PDF-` we check the beginning of PDF
             * https://stackoverflow.com/questions/40376772/using-php-how-i-check-a-pdf-file-contents-is-valid-or-invalid
             */
            if (!preg_match("/^%PDF-/", $filecontent)) {
                throw new \Exception(sprintf('File Key validation failed for host: `%s`. Please reach out to the FileCloud support team at support@filecloud.com', $siteHost));
            }

            $fileKeyVerificationValid++;

            // remove tmp file
            @unlink($srcLocalPath);
        }

        if ($fileKeyVerificationValid === 0) {
            throw new \Exception(sprintf('File Key validation failed for host : `%s`. Please reach out to the FileCloud support team at support@filecloud.com', $siteHost));
        }

        return true;
    }

    private function backupRawFileKey(string $siteHost, string $rawFileKey): bool
    {
        $isMultiTenant = defined("TONIDOCLOUD_MULTISITE_ENABLE") && (int)constant('TONIDOCLOUD_MULTISITE_ENABLE') === 1;

        if ($isMultiTenant) {
            setSiteHostName($siteHost);
        }

        $cryptCollection = StorageConfig::getInstance()->getDBConnection()->selectCollection('cryptfs');
        $keys = $cryptCollection->findOne(['name' => 'keys']);

        if (!isset($keys)) {
            throw new \Exception(
                sprintf(
                    'Cannot backup raw File Key, keys are not found in database and storage encryption is not enabled for host `%s`.',
                    $siteHost
                )
            );
        }

        $newValue = array_merge($keys['value'], ['backupFileKey' => base64_encode($rawFileKey)]);

        $criteria = ['_id' => $keys['_id']];
        $newdata = [
            '$set' => [
                'value' => $newValue
            ]
        ];

        $result = $cryptCollection->updateOne($criteria, $newdata);

        if (!$result) {
            throw new \Exception(sprintf('Problem with saving to db File Key for host `%s`', $siteHost));
        }

        return (bool)$result;
    }

    /**
     * @return array|SiteSetting[]
     */
    private function getAllSites(): array
    {
        $sites = [];
        $siteCursor = $this->settingsDataStore->getSiteSettings();

        if ($siteCursor === null) {
            throw new \Exception('Database Error: Database is not Initialized, please check if database is running.');
        }

        foreach ($siteCursor as $siteItem) {
            $siteSetting = SiteSetting::withDBData($siteItem);

            $sites[] = $siteSetting;
        }

        return $sites;
    }

    private function isDatabaseInitialized(): bool
    {
        $dbServer = 'mongodb://localhost:27017';
        if (defined('TONIDOCLOUD_SETTINGS_DBSERVER')) {
            $dbServer = constant('TONIDOCLOUD_SETTINGS_DBSERVER');
        }

        $conn = Utility::getMongoConnection($dbServer);

        $dbName = 'tonidosettings';
        $dbConn = $conn->selectDB($dbName);

        $pingCursor = $dbConn->command(['ping' => true]);

        $result = $pingCursor->toArray();

        if (isset($result[0]['ok']) && (int)$result[0]['ok'] === 1) {
            return true;
        }

        return false;
    }

    private function isLanManagedStorage(): bool
    {
        return $this->settingsManager->isLocalStorage();
    }

    private function isEncryptedStorageEnabled(): bool
    {
        return StorageConfig::getStorageImpl() instanceof CryptFSStoreImpl;
    }

    private function isDataStoreReady(): bool
    {
        return $this->dataStore->isDataStoreReady();
    }

    private function getRawFileKeyFromMemcache(): ?string
    {
        return CryptHelper::getFileKey();
    }
}