<?php
/* * *****************************************************************************
  Copyrigh(c) 2020 CodeLathe LLC. All rights Reserved.
  This file is part of FileCloud  https://www.filecloud.com
 * ***************************************************************************** */

use core\framework\Utility;
use core\framework\BackgroundDataStore;
use core\framework\BackgroundPayload;
use core\framework\MqDefs;
use core\framework\AuthKey;
use core\framework\TonidoCloudServer;

class MqDrainer
{

    /**
     * This is the main entry point function to performing background processing
     * This is called by MQ worker.
     *
     * Execution time limits are disabled for this function.
     *
     * This function will not return.
     */
    public function processWorkRequest(): void
    {
        // Disable any timeout
        set_time_limit(0);
        global $g_REQUEST;
        global $g_log;

        // Processes upto these many messages to avoid locking up the
        // the Message queue worker waiting for response for too long
        $count = 0;


        // This g_REQUEST is setup before we override it with passed
        // in values. This is coming from Message queue
        if ( empty($g_REQUEST["workerid"])  ||
            empty($g_REQUEST["workertype"]) ||
            empty($g_REQUEST["host"]))
        {
            $g_log->logError(__FUNCTION__ . " Malformed request. Rejecting");
            echo "FAIL";
            exit(0);
        }

        $workerId = $g_REQUEST["workerid"];
        $workerType = $g_REQUEST["workertype"];
        $host = $g_REQUEST["host"];

        $g_log->logDeveloper("BACKGROUND : [$workerId][$workerType] Host=$host");
        setSiteHostName($host);

        $payload = $this->getBackgroundPayload($workerType);
        while ($payload != NULL ) {
            try {
                // Execute Payload
                if ((Utility::startsWith($payload->getBgClass(), "core\\framework\\")) ||
                    (Utility::startsWith($payload->getBgClass(), "app\\"))) {
                    $classname = $payload->getBgClass();
                } else {
                    $classname = "\\core\\framework\\" . $payload->getBgClass();
                }

                $obj = new $classname();
                $params = $payload->getApiParam();


                $this->setupEnv($params);

                $g_log->logDebug("BACKGROUND : [$workerId][$workerType] Calling : "
                                 . $payload->getBgClass() . ":" . $payload->getBgMethod());
                $this->execute($obj, $payload->getBgMethod(), $params, $workerType, $workerId);

                $count ++;

                // Ideally we should not have any more process. But if for some reason we have background
                // stuff it needs to be processed
                $payload = $this->getBackgroundPayload($workerType);
                if (!$payload) {
                    // Wait a bit in case of processes that may still queue items
                    sleep(3);
                    $payload = $this->getBackgroundPayload($workerType);
                }
            } catch (\Throwable $t) {
                // Do Nothing
                $g_log->logDebug("BACKGROUND :  Caught exception " . $t->getMessage());
            }
        }

        echo sprintf("BACKGROUND : %s draining complete with %d items\n", $workerType, $count);
    }

    /**
     * This method will drain one item out of the queue. Depending
     * on the worker type, Serial or Parallel queue will be drained.
     * @param WorkerType: Serial or Parallel
     * @return BackgroundPayload or NULL
     */
    private function getBackgroundPayload($workerType): ?BackgroundPayload
    {
        $qdb = new BackgroundDataStore();

        $parallel = false;
        if ($workerType == MqDefs::PARALLEL) {
            $parallel = true;
        }

        $bgp = NULL;

        if ($parallel) {
            $csm = new \core\framework\CriticalSectionManager();
            //$g_log->logDebug("[$workerId] Acquire section");
            if ($csm->acquireSection("PARALLEL_MQ", 10, 11, false)) {
                $bgp = $qdb->getBackgroundPayload($parallel);
                //$g_log->logDebug("[$workerId] Release section");
                $csm->releaseSection("PARALLEL_MQ");
            }
        } else {
            $bgp = $qdb->getBackgroundPayload($parallel);
        }

        return $bgp;
    }

    /**
     * Setup environment before executing callback
     *
     * This includes setting up globals and auth key based
     * on the data in the parameter list
     *
     * @global type $g_contextArray
     * @global type $g_REQUEST
     * @param type $params
     */
    private function setupEnv($params) : void
    {
        global $g_contextArray;
        global $g_REQUEST;
        global $g_log;

        // Get the global context and set it before calling function
        if (isset($params["globalcontext"])) {
            $g_contextArray = $params["globalcontext"];
        }

        // If g_REQUEST data was sent, unserialize it here
        if (isset($params["grequest"])) {
            $g_REQUEST = unserialize($params["grequest"]);
            //$g_log->logDebug("Setting gREQUEST = " . print_r($g_REQUEST,true));
        }

        // In case anybody in this process queue wants to know if we are
        // in background mode. This will be useful
        setGlobalContext("BACKGROUNDMODE", true);

        //$g_log->logDebug(__FUNCTION__ . " : " . print_r($params,true));

        // If authkey is sent, then set that here before calling the user function
        if (isset($params['role'])) {
            $authKey = new AuthKey($params['role'], $params['source'], $params['context'], $params['session']);

            if (isset($authKey)) {
                //$g_log->logDebug("BACKGROUND : [$workerId][$workerType] Setting auth as "
                //    . "[".$params['context'] ."] and Role as [".$params['role']."]" );
                TonidoCloudServer::getInstance()->setAuthKeyFromKey($authKey);
            }
        }
    }

    /**
     * This is the actual execution of the background callback.
     *
     * For serial processing, csec is required to make sure we ensure
     * serialized processing in multi node environment
     *
     *
     * @param Object $object : The class to call
     * @param String $method : The method to call
     * @param Array $param : The parameter array to pass
     * @param String $workerType : Type of execution. Serial or parallel
     * @param String $workerId : Worker id for tracking purposes
     * @return void
     */
    private function execute($object, $method, $param, $workerType, $workerId): void
    {
        // Serial processing has to take critical section
        // to ensure serialization in HA environment.
        global $g_log;
        $gotLock = false;
        $csm = new \core\framework\CriticalSectionManager();
        if ($workerType == MqDefs::SERIAL) {

            // Critical section. Call can take upto 5 minutes and wait for
            // a little longer than that. This time is arbitarily selected. None
            // of these calls are expected to take very long
            if (!$csm->acquireSection("SERIAL_BACKGROUND_PROCESS", 300, 350, false)) {
                // Couldnt get lock. It is ok. The queue will be processed.
                return;
            }
            $gotLock = true;
        }

        try {
            /******************* CALLBACK ***************************/
            call_user_func(array($object, $method), $param);
            /********************************************************/
        } catch (\Exception $e) {
            $g_log->logError("BACKGROUND : [$workerId][$workerType] " .
                             get_class($object) . ":" .
                             $method . " : Failed with exception: " .
                             $e->__toString());
        }

        // Release the lock if we got it
        if ($gotLock) {
            $csm->releaseSection("SERIAL_BACKGROUND_PROCESS");
        }
    }
}
