<?php

/**
 * Copyright (c) 2020 CodeLathe. All rights Reserved.
 * This file is part of FileCloud  https://www.filecloud.com
 */

namespace app\backupserver;

require_once TONIDO_CLOUD_ROOT_DIR . DIRECTORY_SEPARATOR . 'src' . DIRECTORY_SEPARATOR . 'runtime.php';

if (is_readable(TONIDOCLOUD_CONFIG_PATH . DIRECTORY_SEPARATOR . 'backupserverconfig.php')) {
    require_once TONIDOCLOUD_CONFIG_PATH . DIRECTORY_SEPARATOR . 'backupserverconfig.php';
}

require_once 'backuptarget.class.php';
require_once 'backupjob.class.php';
require_once 'backuputils.class.php';

use CodeLathe\Application\LoggerFacade;
use CodeLathe\Core\Common\Utility\SanitizeUtility;
use CodeLathe\Core\Common\Utility\Utility;
use CodeLathe\Core\Subsystem\Admin\AdminManager;

//For security, backupserver has to be explicitly enabled
if (!Utility::isBackupServerAppEnabled()) {
    LoggerFacade::error('Backupserver application not enabled');
    Utility::getXMLResult('op', '0', 'Backupserver application not enabled');
    exit;
}

$op = SanitizeUtility::sanitizeString((string)filter_input(INPUT_POST, 'op'));
if (empty($op)) {
    $op = SanitizeUtility::sanitizeString((string)filter_input(INPUT_GET, 'op'));
}
if (empty($op)) {
    BackupUtils::sendXMLError('Missing OP');
}

if (!Utility::isValidInput($op)) {
    Utility::getXMLResult('op', '0', 'Invalid OP');
    exit;
}

$op = Utility::sanitizeString($_REQUEST['op']);

Utility::logEntry($op);
BackupUtils::checkauth();

/* Process Request */
switch ($op) {
    //getconfigsetting call has to be routed through here because
    //the backupserver config is read only here
    case 'getconfigsetting':
        $obj = new AdminManager();
        $obj->getConfigSetting();

        break;
    case 'checkpath':
        BackupUtils::check_path();

        break;
    case 'versioninfo':
        BackupUtils::version();

        break;
    //----------Backup target specific calls-----------------
    case 'addtarget':
        $target = new BackupTarget();
        $target->create();

        break;

    case 'deletetarget':
        $target = new BackupTarget();
        $target->delete();

        break;

    case 'updatetarget':
        $target = new BackupTarget();
        $target->update();

        break;

    case 'listtargets':
        $target = new BackupTarget();
        $target->getlist();

        break;

    //----------Backup job specific calls-----------------
    case 'addjob':
        $job = new BackupJob();
        $job->create();

        break;

    case 'deletejob':
        set_time_limit(0);
        $job = new BackupJob();
        $job->delete_fromui();

        break;

    case 'listjobs':
        $job = new BackupJob();
        $job->getlist();

        break;

    case 'startbackupjob':
        $job = new BackupJob();
        $job->startBackupJob();

        break;

    case 'startrestorejob':
        $job = new BackupJob();
        $job->startRestoreJob();

        break;

    case 'jobstatus':
        $job = new BackupJob();
        $job->getjobstatus();

        break;
    case 'requestjobcancel':
        $job = new BackupJob();
        $job->requestcancel();

        break;

    //----------Backed up file specific calls-----------------
    case 'getfilelist':
        $bsw = new BackupStorageWrapper();
        $bsw->getFileList();

        break;
    case 'downloadfile':
        $bsw = new BackupStorageWrapper();
        $bsw->downloadFile();

        break;
    case 'downloadfilemulti':
        $bsw = new BackupStorageWrapper();
        $bsw->downloadFileMulti();

        break;

    //----------File/Folder restore calls-----------------
    case 'addtorestorequeue':
        $bsw = new BackupStorageWrapper();
        $bsw->addToUploadQueue();

        break;
    case 'delfromrestorequeue':
        $bsw = new BackupStorageWrapper();
        $bsw->removeFromUploadQueue();

        break;
    case 'getuploadqueue':
        $bsw = new BackupStorageWrapper();
        $bsw->getUploadQueue();

        break;

    //----------File/Folder export calls-----------------
    case 'addtoexportqueue':
        $bsw = new BackupStorageWrapper();
        $bsw->addToExportQueue();

        break;
    case 'delfromexportqueue':
        $bsw = new BackupStorageWrapper();
        $bsw->removeFromExportQueue();

        break;
    case 'getexportqueue':
        $bsw = new BackupStorageWrapper();
        $bsw->getExportQueue();

        break;
    case 'startexportjob':
        $job = new BackupJob();
        $job->startExportJob();

        break;
    case 'startexportjobdebug':
        //used for debug purposes
        $job = new BackupJob();
        $job_db_id = SanitizeUtility::sanitizeString((string)filter_input(INPUT_GET, '_id'));
        $job->export($job_db_id);

        break;
    default:
        Utility::getXMLResult('op', '0', 'Unsupported OP');
        exit;
}
