<?php

/**
 * Copyright (c) 2020 CodeLathe. All rights Reserved.
 * This file is part of FileCloud  https://www.filecloud.com
 */

/*
    SIEM AUDIT MAPPING ONLY
    Notes: Please adjust accordingly. More information can be found in the official SIEM documentation
    Those are sample mappings and should be treated as a guidelines only.
    Severity has to be adjusted based on the internal rules.
*/

$mappings = [];

/**************************************** Downloads ****************************************/
// Download file
$mappings[] = [
    'id' => 'downloadfile',
    'prefilter' => [],
    'map' => [
        'eventClass' => 'FileOperations',
        'eventName' => '$operation',
        'severity' => 2,
        'extension' => [
            'suser' => '$userName',
            'shost' => '$hostname', // name of the host
            'recordId' => '$recordId', // Audit record id
            'requestClientApplication' => '$userAgent',
            'src' => '$ip',
            'fname' => '$request.filename > $notes', // $notes is a fallback for downloadfilemulti operation
            'filePath' => '$realpath > $request.filePath', // realpath is used for downloadfilemulti
            'fsize' => '$bandwidth',
            'cs1' => '$metadata',
            'cs1Label' => 'Metadata assigned to the file'
        ]
    ]
];

/***************************************** Uploads *****************************************/
// Upload
$mappings[] = [
    'id' => 'upload',
    'prefilter' => [],
    'map' => [
        'eventClass' => 'FileOperations',
        'eventName' => '$operation',
        'severity' => 2,
        'extension' => [
            'suser' => '$userName',
            'shost' => '$hostname', // name of the host
            'recordId' => '$recordId', // Audit record id
            'requestClientApplication' => '$userAgent',
            'src' => '$ip',
            'fname' => '$request.filename', // $notes can be used as well
            'filePath' => '$request.path',
            'fsize' => '$bandwidth'
        ]
    ]
];

/****************************************** Shares *****************************************/
// addusertoshare - Adding user to the existing share
$mappings[] = [
    'id' => 'addusertoshare',
    'prefilter' => [],
    'map' => [
        'eventClass' => 'Shares',
        'eventName' => '$operation',
        'severity' => 2,
        'extension' => [
            'suser' => '$userName',
            'shost' => '$hostname', // name of the host
            'recordId' => '$recordId', // Audit record id
            'requestClientApplication' => '$userAgent',
            'src' => '$ip',
            'filePath' => '$notes',
            'duser' => '$request.userid',
            'cs1' => '$metadata',
            'cs1Label' => 'Metadata assigned to the file'
        ]
    ]
];

// updateshare - updating existing share
$mappings[] = [
    'id' => 'updateshare',
    'prefilter' => [],
    'map' => [
        'eventClass' => 'Shares',
        'eventName' => '$operation',
        'severity' => 2,
        'extension' => [
            'suser' => '$userName',
            'shost' => '$hostname', // name of the host
            'recordId' => '$recordId', // Audit record id
            'requestClientApplication' => '$userAgent',
            'src' => '$ip',
            'filePath' => '$request.sharelocation',
            'cs1' => '$metadata',
            'cs1Label' => 'Metadata assigned to the file'
        ]
    ]
];

// setuseraccessforshare - sets user permissions for share
$mappings[] = [
    'id' => 'setuseraccessforshare',
    'prefilter' => [],
    'map' => [
        'eventClass' => 'Shares',
        'eventName' => '$operation',
        'severity' => 6, // this can be a potentially risky operation since data exposure and leakage might happen
        'extension' => [
            'suser' => '$userName',
            'shost' => '$hostname', // name of the host
            'recordId' => '$recordId', // Audit record id
            'requestClientApplication' => '$userAgent',
            'src' => '$ip',
            'filePath' => '$notes',
            'duser' => '$request.userid',
            'cs1' => '$metadata',
            'cs1Label' => 'Metadata assigned to the file',
            'cs2' => '$request.shareid',
            'cs2Label' => 'Share Identifier'
        ]
    ]
];

// setallowpublicaccess - happens when a share is mad public
$mappings[] = [
    'id' => 'setallowpublicaccess',
    'prefilter' => [],
    'map' => [
        'eventClass' => 'Shares',
        'eventName' => '$operation',
        'severity' => 6, // this can be a potentially risky operation since data exposure and leakage might happen
        'extension' => [
            'suser' => '$userName',
            'shost' => '$hostname', // name of the host
            'recordId' => '$recordId', // Audit record id
            'requestClientApplication' => '$userAgent',
            'src' => '$ip',
            'filePath' => '$notes',
            'ispublic' => '$request.allowpublicaccess', // 1 - public share, 0 - private share
            'cs1' => '$metadata',
            'cs1Label' => 'Metadata assigned to the file',
            'cs2' => '$request.shareid',
            'cs2Label' => 'Share Identifier'
        ]
    ]
];

/**************************************** Smart DLP ****************************************/
// DLP Violation
$mappings[] = [
    'id' => 'dlp',
    'prefilter' => [],
    'map' => [
        'eventClass' => 'DLP Violation',
        'eventName' => '$operation',
        'severity' => 6,
        'extension' => [
            'suser' => '$userName',
            'shost' => '$hostname', // name of the host
            'recordId' => '$recordId', // Audit record id
            'requestClientApplication' => '$userAgent',
            'src' => '$ip',
            'filePath' => '$realpath',
            'msg' => '$notes.message',
            'shareTargetEmail' => '$notes.shareTargetEmail',
            'cs1' => '$metadata',
            'cs1Label' => 'Metadata assigned to the file',
            'cs3' => '$request.op',
            // operation that triggered the violation / $notes.action can be uses as well for a less granular info: DOWNLOAD / SHARE / LOGIN
            'cs3Label' => 'DLP Violation trigger',
            // Additional information can be grabbed from the request object
            'cs4' => '$notes.violatedRule',
            // DLP rule that was violated
            'cs4Label' => 'DLP Violation rule'
        ]
    ]
];

/*********************************** Smart Classification **********************************/
// Smart Classification - apply match action
$mappings[] = [
    'id' => 'ccsapplymatchaction',
    'prefilter' => [],
    'map' => [
        'eventClass' => 'CCE match',
        'eventName' => '$operation',
        'severity' => 2,
        'extension' => [
            'suser' => '$userName',
            'shost' => '$hostname', // name of the host
            'recordId' => '$recordId', // Audit record id
            'requestClientApplication' => '$userAgent',
            'src' => '$ip',
            'msg' => '$notes'
        ]
    ]
];

/************************************* AV - Virus removed ***********************************/
// When AV finds and removes the file containing a Virus (i.e. ICAP AV)
$mappings[] = [
    'id' => 'virusremoved',
    'prefilter' => [],
    'map' => [
        'eventClass' => 'virusremoved',
        'eventName' => 'Virus Removed',
        'severity' => 8,
        'extension' => [
            'user' => '$userName',
            'shost' => '$hostname', // name of the host
            'recordId' => '$recordId', // Audit record id
            'userAgent' => '$userAgent',
            'ip' => '$ip',
            'fname' => '$request.filename',
            'filePath' => '$request.path',
            'notes' => '$notes'
        ]
    ]
];

/******************************* Group management *****************************************/

// Group rename
$mappings[] = [
    'id' => 'updategroup',
    'prefilter' => [],
    'map' => [
        'eventClass' => 'Groups',
        'eventName' => '$operation',
        'severity' => 6,
        'extension' => [
            'suser' => '$userName',
            'shost' => '$hostname', // name of the host
            'recordId' => '$recordId', // Audit record id
            'requestClientApplication' => '$userAgent',
            'src' => '$ip',
            'msg' => '$notes'
        ]
    ]
];

$mappings[] = [
    'id' => 'addmembertogroup',
    'prefilter' => [],
    'map' => [
        'eventClass' => 'Groups',
        'eventName' => '$operation',
        'severity' => 5,
        'extension' => [
            'suser' => '$userName',
            'shost' => '$hostname', // name of the host
            'recordId' => '$recordId', // Audit record id
            'requestClientApplication' => '$userAgent',
            'src' => '$ip',
            'duser' => '$request.userid',
            'msg' => '$notes'
        ]
    ]
];

$mappings[] = [
    'id' => 'deletememberfromgroup',
    'prefilter' => [],
    'map' => [
        'eventClass' => 'Groups',
        'eventName' => '$operation',
        'severity' => 5,
        'extension' => [
            'suser' => '$userName',
            'shost' => '$hostname', // name of the host
            'recordId' => '$recordId', // Audit record id
            'requestClientApplication' => '$userAgent',
            'src' => '$ip',
            'duser' => '$request.userid',
            'msg' => '$notes'
        ]
    ]
];

/******************************* User management *****************************************/

$mappings[] = [
    'id' => 'adduser',
    'prefilter' => [],
    'map' => [
        'eventClass' => 'Users',
        'eventName' => '$operation',
        'severity' => 5,
        'extension' => [
            'suser' => '$userName',
            'shost' => '$hostname', // name of the host
            'recordId' => '$recordId', // Audit record id
            'requestClientApplication' => '$userAgent',
            'src' => '$ip',
            'duser' => '$request.username', // name of the user that has been added
            'msg' => '$notes' // More info about the user
        ]
    ]
];

// Admin status change
$mappings[] = [
    'id' => 'setadminstatus',
    'prefilter' => [],
    'map' => [
        'eventClass' => 'Users',
        'eventName' => '$operation',
        'severity' => 2,
        'extension' => [
            'suser' => '$userName',
            'shost' => '$hostname', // name of the host
            'recordId' => '$recordId', // Audit record id
            'requestClientApplication' => '$userAgent',
            'src' => '$ip',
            'duser' => '$request.profile',
            'msg' => '$request.adminstatus'
        ]
    ]
];

// User password changed by admin
$mappings[] = [
    'id' => 'setuserpassword',
    'prefilter' => [],
    'map' => [
        'eventClass' => 'Users',
        'eventName' => '$operation',
        'severity' => 2,
        'extension' => [
            'suser' => '$userName', // Admin who performed the operation
            'shost' => '$hostname', // name of the host
            'recordId' => '$recordId', // Audit record id
            'requestClientApplication' => '$userAgent',
            'src' => '$ip',
            'duser' => '$request.profile' // User whose password has been changed
        ]
    ]
];

/****************************************** Generic ****************************************/
// Please uncomment to use a generic mapper

//$mappings[] = [
//    'id' => '*',
//    'prefilter' => [],
//    'map' => [
//        'eventClass' => '$operation',
//        'eventName' => '$operation',
//        'severity' => 2,
//        'extension' => [
//            'suser' => '$userName',
//            'requestClientApplication' => '$userAgent',
//            'src' => '$ip',
//            'msg' => '$notes'
//        ]
//    ]
//];

return $mappings;
