<?php

/*
 * Copyright (c) 2021 CodeLathe. All rights Reserved.
 * This file is part of FileCloud  https://www.filecloud.com
 */


namespace CodeLathe\Core\Common\Utility;

use CodeLathe\Application\GlobalStore;
use CodeLathe\Core\Subsystem\Integration\MSTeams\MSTeamsManager;

class CookieUtility
{
    public const CONTEXT_COOKIE_NAME = 'filecloud-context';
    public const SALESFORCE_CONTEXT_COOKIE_NAME = 'Salesforce';

    /**
     * Checks whether Same-Site: None; Secure; header is required for Cookies. This might be needed for iframe
     * integrations like Salesforce or MS Teams
     * @return bool
     */
    public static function isSameSiteNoneRequired(): bool
    {
        $contextCookie = self::getContextCookie();
        return in_array($contextCookie, [self::SALESFORCE_CONTEXT_COOKIE_NAME]) ||
            MSTeamsManager::isMSTeamsRequest(GlobalStore::getCurrentRequest());
    }

    /**
     * Returns the value for the filecloud-context cookie
     * @return string
     */
    public static function getContextCookie(): string
    {
        return $_COOKIE[self::CONTEXT_COOKIE_NAME] ?? '';
    }

    /**
     * Generates the Salesforce context cookie
     */
    public static function setSalesforceContextCookieHeader()
    {
        Utility::setCookie(self::CONTEXT_COOKIE_NAME, 'Salesforce', 0, true);
    }

    /**
     * A temporary wrapper method that extracts the setcookie method from the actual Utility::setCookie.
     * The reason for that is our codebase cannot be fully migrated to PHP 7.3+ due the ioncube's limitation.
     * Since our runtime is using PHP 7.4 already (ioncube provides the PHP 7.4 loader) the workaround provided
     * in PHP 7.2 does not work anymore and raises a Warning. The solution is to keep this file unencoded so the
     * new PHP 7.3's setcookie syntax can be used.
     *
     * IMPORTANT: CookieUtility has to be unencoded. This can be set in the ioncube_fcencode.bat script file.
     */
    public static function setCookie(string $name, ?string $value, array $cookieOptions)
    {
        setcookie($name, (string)$value, $cookieOptions);
    }

    /**
     * @param string $sameSiteType
     * @return bool
     */
    public static function isValidSameSiteType(string $sameSiteType): bool
    {
        return in_array($sameSiteType, ['Strict', 'Lax', 'None']);
    }
}
