/*
 * Copyright (c) 2024 FileCloud. All rights Reserved.
 * This file is part of FileCloud https://www.filecloud.com
 */

const { spawn } = require('child_process');
const path = require('path');
const os = require('os');
const fs = require('fs');

const isWin = process.platform === "win32";

// load bootstrap config to find node modules directory
if (fs.existsSync('./config/default.json')) {
    let bootstrapdata = fs.readFileSync('./config/default.json');
    let bootstrapconfig = JSON.parse(bootstrapdata);
    let nodejsmodulespath = "/opt/fcnodejs/node_modules";
    if (isWin) {
        nodejsmodulespath = "..\\..\\..\\nodejs\\node_modules";
    }
    if(bootstrapconfig !== null && bootstrapconfig.nodejsmodules !== ''){
        nodejsmodulespath = bootstrapconfig.nodejsmodules;
    }
    console.log("Using node module path "+ nodejsmodulespath);
    process.env.NODE_PATH = nodejsmodulespath;
    require("module").Module._initPaths();
}

// load non-native modules (node_modules) past this point
const config = require('config');
const logger = require('./logger');

const useGlobalWorkers = config.get("global_worker");
let siteWorkerLimit = config.get('site_worker_limit');
let parallelWorkerCount = config.get('parallel_workers_count');
let parallelHighPriorityWorkerCount = config.get('parallel_high_priority_workers_count');

if (siteWorkerLimit === '') {
    siteWorkerLimit = 10;
}

if (parallelWorkerCount === '') {
    parallelWorkerCount = 4;
}

if (parallelHighPriorityWorkerCount === '') {
    parallelHighPriorityWorkerCount = 0;
}

const phpOverride = config.get("php");
let php = "/usr/bin/php";
if (isWin) {
    php = '..\\..\\..\\php\\php.exe';
}

if (phpOverride !== '') {
    php = phpOverride;
}

if (!fs.existsSync(php)) {
    console.error("CRITICAL: PHP executable [" + php + "] not found!");
    process.exit(1);
}


let launchedParallelSites = [];
let launchedSerialSites = [];
const pidMap = new Map();

/**
 * Gets all the sites in the system
 * @returns {Promise<unknown>}
 */
function getSites() {

    return new Promise((resolve, reject) => {

        try {
            //logger.info("Getting installed sites");
            let sites = [];
            let scriptOutput = '';
            let params =  [path.resolve('fcorchutility.php') , '-s'];
            //logger.info(params);
            let util = spawn(php, params);

            util.stdout.on('data', (data) => {
                //logger.info(data.toString());
                if (data.toString().startsWith('DATA:')) {
                    scriptOutput += data.toString();
                }
            });

            util.on('close', (code) => {
                // Split the strings on carriabe return and string the leading DATA: string
                //logger.info (scriptOutput);
                sites = scriptOutput.split(/\r\n|\n+|\r+|DATA:|;+|$/g).filter(x => x);
                resolve(sites)
            });

        } catch(err) {
            reject(err);
        }
    });
}

/**
 * Launch specific PHP worker to service message queues
 * @param site String with site id
 * @param mode String - SERIAL or PARALLEL
 */
function  launchWorker(site, mode, globalWorker=false) {
    let params =  [path.resolve('bgworker.php') , '-m'+mode];

    if (globalWorker) {
        params.push('-g');
    }
    else {
        params.push('-s' + site);
    }

    //logger.info(params);
    let worker = spawn(php, params);

    pidMap.set(worker.pid, {site:site, mode:mode, globalWorker:globalWorker});

    let siteN  = site;
	logger.info("Spawning " + mode + " worker for monitoring " + ((site)?site:'ALL SITES'));
    worker.on("exit", (code, signal) => {
        logger.info(mode + " worker for " + site + "  exited with " + code + " and signal " + signal + " - Restarting");
        setTimeout(() => {
            if (pidMap.get(worker.pid) != null){
                let o = pidMap.get(worker.pid);
                pidMap.delete(worker.pid);
                launchWorker(o.site, o.mode, o.globalWorker);
            }
        }, 1000)

    });

    worker.stdout.on('data', data => {
        logger.info(`stdout: ${data}`);
    });

    worker.stderr.on('data', data => {
        logger.info(`stderr: ${data}`);
    });
}

function launchSerialWorker(site = '', globalWorker = false) {
    let hostname = os.hostname();

    let params =  [path.resolve('fcorchutility.php') , '-h'+hostname];
    if(site != ''){
        params.push('-n'+site);
    }
    if (globalWorker) {
        params.push('-g');
    }
    //logger.info(params);
    let util = spawn(php, params);

    util.stdout.on('data', (data) => {
        logger.info(data.toString());
    });

    util.on('close', (code) => {
        // Split the strings on carriage return and string the leading DATA: string
        //logger.info (scriptOutput);
        if (code == 0) {
            // Nothing to be done. Dont launch Serial queue, Clear our launched queue
            launchedSerialSites = [];
        }
        else if (code == 1) {
            // We are the serial worker node - Launch
            if (site == '')  {
                if (!launchedSerialSites.includes('GLOBAL')) {
                    launchedSerialSites.push('GLOBAL');
                    launchWorker('', 'SERIAL', globalWorker)

                }
            }
            else {
                if (!launchedSerialSites.includes(site)) {
                    launchedSerialSites.push(site);
                    launchWorker(site, 'SERIAL', globalWorker)
                }
            }
        }
    });
}

function launchSiteSpecificWorkers(sites) {
    for (let site of sites) {
        launchSerialWorker(site);
        if (!launchedParallelSites.includes(site)) {

            // Remember this
            launchedParallelSites.push(site);
            for (let a=0; a<parallelWorkerCount; a++) {
                launchWorker(site, 'PARALLEL');
            }
            for (let a=0; a<parallelHighPriorityWorkerCount; a++) {
                launchWorker(site, 'PARALLEL_HIGH_PRIORITY');
            }
        }
    }
}

function launchGlobalWorkers() {
    launchSerialWorker('', true);

    if (!launchedParallelSites.includes('GLOBAL')) {
        launchedParallelSites.push('GLOBAL');

        for (let a=0; a<parallelWorkerCount; a++) {
            launchWorker('', 'PARALLEL', true);
        }
        for (let a=0; a<parallelHighPriorityWorkerCount; a++) {
            launchWorker(site, 'PARALLEL_HIGH_PRIORITY');
        }
    }
}

async function launchMQWorkers() {
    const sites = await getSites();
    if (useGlobalWorkers === true) {
        launchGlobalWorkers();
    }
    else if (useGlobalWorkers === false) {
        launchSiteSpecificWorkers(sites);
    }
    else if (sites.length <= siteWorkerLimit) {
        //logger.info(sites);
        launchSiteSpecificWorkers(sites);
    } else {
        launchGlobalWorkers();
    }

}


function checkForMqExit() {

    //logger.info('Checking for worker exit request');
    let params =  [path.resolve('fcorchutility.php') , '-m'];
    //logger.info(params);
    let util = spawn(php, params);

    util.stdout.on('data', (data) => {
        logger.info(data.toString());
    });

    util.on('close', (code) => {
        // Split the strings on carriage return and string the leading DATA: string
        //logger.info (scriptOutput);
       if (code === 1) {
            logger.info('Worker restart request received');
            pidMap.forEach(function(value, key) {
                logger.info('Requesting restart for pid ' + key);
                let params =  [path.resolve('fcorchutility.php') , '-r'+key];
                let util2 = spawn(php, params);

                util2.stdout.on('data', (data) => {
                    logger.info(data.toString());
                });

            })

        }
    });
}


launchMQWorkers();

// Keep checking every 1 minutes for any new site
setInterval (() => {
    launchMQWorkers();
}, 60 * 1000);

// Check every 30s for exit command
setInterval (() => {
    checkForMqExit();
}, 30 * 1000);








