<?php

namespace SimpleSAML\Metadata;

use SimpleSAML\Configuration;
use SimpleSAML\Error\Exception;
use SimpleSAML\Logger;

/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

/**
 * Description of SimpleSAML_MetaDataStorageHandlerXMLString
 *
 * @author sathya
 */
class MetaDataStorageHandlerXMLString extends MetaDataStorageSource
{
    /**
     * This variable contains an associative array with the parsed metadata.
     */
    private $metadata;

    private $xmlvalue;


    /**
     * This function initializes the XML metadata source. The configuration must contain one of
     * the following options:
     * - 'xmlvalue': string in xml format.
     *
     * @param array $config The configuration for this instance of the XML metadata source.
     */
    protected function __construct(array $config)
    {
        Logger::info("metadata storage handler xml string");
        if (array_key_exists('xmlvalue', $config)) {
            $this->xmlvalue = $config['xmlvalue'];
        } else {
            Logger::error("Missing Idp Metadata xml string value in configuration.");
            throw new Exception('Missing either xml string value in XML metadata source configuration.');
        }

        $SP1x = [];
        $IdP1x = [];
        $SP20 = [];
        $IdP20 = [];
        $AAD = [];

        $entities = $this->parseDescriptorsXmlString();
        foreach ($entities as $entityId => $entity) {
            /**
             * @info getMetadata1xSP and getMetadata1xIdP methods no longer exists in server/thirdparty/simplesaml/src/SimpleSAML/Metadata/SAMLParser.php
             * as shib13 support has been dropped
             */
//            $md = $entity->getMetadata1xSP();
//            if ($md !== null) {
//                $SP1x[$entityId] = $md;
//            }
//
//            $md = $entity->getMetadata1xIdP();
//            if ($md !== null) {
//                $IdP1x[$entityId] = $md;
//            }

            $md = $entity->getMetadata20SP();
            if ($md !== null) {
                $SP20[$entityId] = $md;
            }

            $md = $entity->getMetadata20IdP();
            if ($md !== null) {
                $IdP20[$entityId] = $md;
            }

            $md = $entity->getAttributeAuthorities();
            if (count($md) > 0) {
                $AAD[$entityId] = $md[0];
            }
        }

        $this->metadata = [
            'shib13-sp-remote' => $SP1x,
            'shib13-idp-remote' => $IdP1x,
            'saml20-sp-remote' => $SP20,
            'saml20-idp-remote' => $IdP20,
            'attributeauthority-remote' => $AAD,
        ];
    }


    /**
     * This function returns an associative array with metadata for all entities in the given set. The
     * key of the array is the entity id.
     *
     * @param string $set The set we want to list metadata for.
     * @return array An associative array with all entities in the given set.
     */
    public function getMetadataSet(string $set): array
    {
        if (array_key_exists($set, $this->metadata)) {
            return $this->metadata[$set];
        }

        /* We don't have this metadata set. */
        return [];
    }


    private function parseDescriptorsXmlString()
    {
        $data = $this->xmlvalue;

        $doc = new \DOMDocument();

        $res = $doc->loadXML($data);

        if ($res !== true) {
            throw new Exception('Failed to read Idp Metadata XML String');
        }

        if ($doc->documentElement === null) {
            throw new Exception('Idp Metadata String is not a valid XML document');
        }

        Logger::info("Idp Metadata is valid xml format");

        return SAMLParser::parseDescriptorsElement($doc->documentElement);
    }

}